/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: CifXDeviceBase.cpp 13048 2019-08-07 13:35:21Z LuisContreras $:

  Description:
    CifX Device base class, for accessing System Device and channel via the same functions

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2019-08-07  Display Interrupt number as a signed number (MSIs are negative numbers)
    2011-12-15  Changed descriptions to "Physical DPM Size"
    2006-06-28  initial version

**************************************************************************************/

#include "StdAfx.h"
#include ".\cifxdevicebase.h"

///////////////////////////////////////////////////////////////////////////////////////////
/// \file CifxDeviceBase.cpp
/// CifX Device base class, for accessing System Device and channel via the same functions
///////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
/// Constructor
///   \param ptBoardInfo    Pointer to board information from driver
///   \param ptChannelInfo  Pointer to channel information from driver
/////////////////////////////////////////////////////////////////////////////
CCifXDeviceBase::CCifXDeviceBase(BOARD_INFORMATION* ptBoardInfo, CHANNEL_INFORMATION* ptChannelInfo)
{
  if(ptBoardInfo)
    m_tBoardInfo = *ptBoardInfo;
  else
    ZeroMemory(&m_tBoardInfo, sizeof(m_tBoardInfo));

  if(ptChannelInfo)
    m_tChannelInfo = *ptChannelInfo;
  else
    ZeroMemory(&m_tChannelInfo, sizeof(m_tChannelInfo));
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CCifXDeviceBase::~CCifXDeviceBase(void)
{
}

/////////////////////////////////////////////////////////////////////////////
/// Get the device specific properties
///   \return Array of all property/value pairs
/////////////////////////////////////////////////////////////////////////////
std::vector<CCifXDeviceBase::PROPERTY_LIST> CCifXDeviceBase::GetDescription(void)
{
  std::vector<CCifXDeviceBase::PROPERTY_LIST> cvList;

  PROPERTY_LIST tList;

  tList.csProperty = _T("Physical Address");
  tList.csValue.Format(_T("0x%08X"), m_tBoardInfo.ulPhysicalAddress);
  cvList.push_back(tList);

  tList.csProperty = _T("Interrupt");
  tList.csValue.Format(_T("%d"), (int8_t)m_tBoardInfo.ulIrqNumber);
  cvList.push_back(tList);

  if(CIFX_NO_ERROR != m_tBoardInfo.lBoardError)
  {
    tList.csProperty = _T("Board Error (Code)");
    tList.csValue.Format(_T("0x%08X"), 
                         m_tBoardInfo.lBoardError);
    cvList.push_back(tList);

    tList.csProperty = _T("Board Error (Descr.)");
    tList.csValue.Format(_T("%s"), 
                         GetErrorDescription(m_tBoardInfo.lBoardError));
    cvList.push_back(tList);

  } else
  {
    tList.csProperty = _T("Device Number");
    tList.csValue.Format(_T("%u"), m_tBoardInfo.tSystemInfo.ulDeviceNumber);
    cvList.push_back(tList);

    tList.csProperty = _T("Serial Number");
    tList.csValue.Format(_T("%u"), m_tBoardInfo.tSystemInfo.ulSerialNumber);
    cvList.push_back(tList);
  }

  tList.csProperty = _T("Physical DPM Size");
  tList.csValue.Format(_T("%u"), m_tBoardInfo.ulDpmTotalSize);
  cvList.push_back(tList);

  return cvList;
}

/////////////////////////////////////////////////////////////////////////////
/// Get Channel's Property/Value pairs for the Channel/Device
///   \return Property/Value pairs (strings)
/////////////////////////////////////////////////////////////////////////////
CCifXDeviceBase::PROPERTY_VECTOR CCifXDeviceBase::GetDriverInformation()
{
  DRIVER_INFORMATION tDriverInfo = {0};

  long lRet = xDriverGetInformation(NULL, 
                                    sizeof(tDriverInfo), 
                                    &tDriverInfo);
  PROPERTY_VECTOR cvRet;

  if(CIFX_NO_ERROR == lRet)
  {
    PROPERTY_LIST tList;

    tList.csProperty = _T("Driver Version");
    tList.csValue.Empty();
    
    for(int iIdx = 0; iIdx < sizeof(tDriverInfo.abDriverVersion) / sizeof(tDriverInfo.abDriverVersion[0]); ++iIdx)
      tList.csValue.AppendFormat(_T("%c"), (char)tDriverInfo.abDriverVersion[iIdx]);
    cvRet.push_back(tList);

    tList.csProperty = _T("Board Count");
    tList.csValue.Format(_T("%u"), tDriverInfo.ulBoardCnt);
    cvRet.push_back(tList);
  }
  
  return cvRet;
}
